class App.Model extends Spine.Model
  @apiPath: App.Config.get('api_path')

  @allowedReplaceTagsFunctionMapping = {}

  constructor: ->
    super

  uiUrl: ->
    '#'

  translate: ->
    App[ @constructor.className ].configure_translate

  objectDisplayName: ->
    @constructor.className

  displayName: ->
    return @name || '-'

  # shows the icon representing the object itself (e. g. the organization icon in organization profile or ticket sidebar)
  icon: (user) ->
    ''

  # shows icons in a list of objects (e. g. the traffic light rings in the ticket list in user profile)
  iconTitle: (user) ->
    ''

  # shows icons in the activity stream (e. g. show ! if the activity stream icon is belonging to the session user)
  iconActivity: (user) ->
    ''

  @validate: (data = {}) ->
    screen = data?.controllerForm?.screen

    if data?.controllerForm?.mixedAttributes
      attributes = data?.controllerForm?.mixedAttributes

    # based on model attributes
    else if App[ data['model'] ] && App[ data['model'] ].attributesGet
      attributes = App[ data['model'] ].attributesGet(screen)

    # based on custom attributes
    else if data['model'].configure_attributes
      attributes = App.Model.attributesGet(screen, data['model'].configure_attributes)

    # check attributes/each attribute of object
    errors = {}
    for attributeName, attribute of attributes

      # only if attribute is not read only
      if !attribute.readonly

        # check required // if null is defined && null is false
        if data.controllerForm && data.controllerForm.attributeIsMandatory(attribute.name)

          # check :: fields
          parts = attribute.name.split '::'
          if parts[0] && !parts[1]

            # key exists not in hash || value is ''|null|undefined
            if !( attributeName of data['params'] ) || @_validate_is_empty(data['params'][attributeName])
              errors[attributeName] = __('is required')

          else if parts[0] && parts[1] && !parts[2]

            # key exists not in hash || value is '' || value is undefined
            if !data.params[parts[0]] || !( parts[1] of data.params[parts[0]] ) || @_validate_is_empty(data.params[parts[0]][parts[1]])
              errors[attributeName] = __('is required')

          else
            throw "can't parse '#{attribute.name}'"

        # check confirm password
        if attribute.type is 'password' && data['params'][attributeName] && "#{attributeName}_confirm" of data['params']

          # get confirm password
          if data['params'][attributeName] isnt data['params']["#{attributeName}_confirm"]
            errors[attributeName] = 'didn\'t match'
            errors["#{attributeName}_confirm"] = ''

        # check email
        if attribute.type is 'email' && data['params'][attributeName]
          if !data['params'][attributeName].match(/^(([^<>()\[\]\.,;:\s@\"]+(\.[^<>()\[\]\.,;:\s@\"]+)*)|(\".+\"))@(([^<>()[\]\.,;:\s@\"]+\.)+[^<>()[\]\.,;:\s@\"]{2,})$/i)
            errors[attributeName] = 'invalid'
          if data['params'][attributeName].match(/ /)
            errors[attributeName] = 'invalid'

        # check datetime
        if attribute.tag is 'datetime'
          if data['params'][attributeName] is 'invalid'
            errors[attributeName] = 'invalid'

          # validate value

        # check date
        if attribute.tag is 'date'
          if data['params'][attributeName] is 'invalid'
            errors[attributeName] = 'invalid'

          # validate value

    if data?.controllerForm && App.FormHandlerCoreWorkflow.requestsRunning(data.controllerForm)
      errors['_core_workflow'] = { target: data.target, controllerForm: data.controllerForm }

    # return error object
    if !_.isEmpty(errors)
      if !errors['_core_workflow']
        App.Log.error('Model', 'validation failed', errors)
      return errors

    # return no errors
    return

  @_validate_is_empty: (value) ->
    return true if value is ''
    return true if value is null
    return true if value is undefined
    return true if _.isArray(value) is true && value.length is 0
    return true if _.isArray(value) is true && value.length is 1 && value[0] is ''
    false

  ###

set new attributes of model (remove already available attributes)

  App.Model.attributesSet(attributes)

  ###

  @attributesSet: (attributes) ->

    configure_attributes = App[ @.className ].configure_attributes
    attributesNew = []
    for localAttribute in configure_attributes
      found = false
      for attribute in attributes
        if attribute.name is localAttribute.name
          found = true
          break
      if !found
        attributesNew.push localAttribute
    for attribute in attributes
      App[@.className].attributes.push attribute.name
      attributesNew.push attribute
    App[@.className].configure_attributes = attributesNew

  ###

  attributes = App.Model.attributesGet(optionalScreen, optionalAttributesList)

  returns
    {
      'name': {
        name:    'name'
        display: __('Name')
        tag:     'input'
        type:    'text'
        limit:   100
        null:    false
      },
      'assignment_timeout': {
        name:    'assignment_timeout'
        display: __('Assignment timeout')
        tag:     'input'
        type:    'text'
        limit:   100
        null:    false
      },
    }

  ###

  @attributesGet: (screen = undefined, attributes = false, noDefaultAttributes = false, className = undefined, renderTarget = undefined) ->
    if !className
      className = @.className

    if !attributes
      attributes = clone(App[className].configure_attributes, true)
    else
      attributes = clone(attributes, true)

    # in case if no configure_attributes exist
    return {} if !attributes

    # check params of screen if screen is requested
    attributesNew = {}
    if screen
      for attribute in attributes
        if attribute && attribute.screen && attribute.screen[screen] && (!_.isEmpty(attribute.screen[screen]) && (attribute.screen[screen].shown is true || attribute.screen[screen].shown is undefined || App.FormHandlerCoreWorkflow.checkScreen(className, screen)))
          for item, value of attribute.screen[screen]
            attribute[item] = value
          attributesNew[ attribute.name ] = attribute

    # if no screen is given or no attribute has this screen - use default attributes
    if (!screen || _.isEmpty(attributesNew)) && !noDefaultAttributes
      for attribute in attributes
        attributesNew[ attribute.name ] = attribute

    for key, attribute of attributesNew
      if renderTarget
        attribute.renderTarget = renderTarget
      if attribute.diff
        current_time = new Date().getTime()
        switch attribute.tag
          when 'date'
            date = new Date(current_time + attribute.diff * 3600 * 1000)
            attribute.default = App.i18n.convert(date, 0, 'yyyy-mm-dd')
          when 'datetime'
            date = new Date(current_time + attribute.diff * 60 * 1000)
            date.setSeconds(0)
            date.setMilliseconds(0)
            attribute.default = date.toISOString()

    attributesNew

  validate: (params = {}) ->
    App.Model.validate(
      model:  @constructor.className
      params: @
      controllerForm: params.controllerForm
      target: params.target
    )

  isOnline: ->
    return false if !@id
    return true if typeof @id is 'number' # in case of real database id
    return true if @id[0] isnt 'c'
    return false

  # App.Model.fullLocal(id)
  @fullLocal: (id) ->
    @_fillUp( App[@className].find(id) )

  # App.Model.full(id, callback, force, bind)
  @full: (id, callback = false, force = false, bind = false) ->

    # skip full requests for guessing users since they won't return results anyway
    # and produce a lot of errors in the log #5132
    return if @className is 'User' && typeof id is 'string' && id.startsWith('guess:')

    url = "#{@url}/#{id}?full=true"

    # subscribe and reload data / fetch new data if triggered
    subscribeId = undefined
    if bind
      subscribeId = App[@className].subscribeItem(id, callback)

    # execute if object already exists
    if !force && App[@className].exists(id)
      data = App[@className].find(id)
      data = @_fillUp(data)
      if callback
        callback(data, 'full')
      return subscribeId

    # store callback and requested id
    if !@FULL_CALLBACK
      @FULL_CALLBACK = {}
    if !@FULL_CALLBACK[id]
      @FULL_CALLBACK[id] = {}
    if callback
      key = @className + '-' + Math.floor( Math.random() * 99999 )
      @FULL_CALLBACK[id][key] = callback

    if !@FULL_FETCH
      @FULL_FETCH = {}
    if !@FULL_FETCH[id]
      @FULL_FETCH[id] = true
      App.Log.debug('Model', "fetch #{@className}.find(#{id}) from server", url)
      App.Ajax.request(
        type:  'GET'
        url:   url
        processData: true,
        success: (data, status, xhr) =>
          @FULL_FETCH[id] = false

          App.Log.debug('Model', "got #{@className}.find(#{id}) from server", data)

          # full / load assets
          if data.assets
            App.Collection.loadAssets(data.assets, targetModel: @className)

          # find / load object
          else
            App[@className].refresh(data)

          # execute callbacks
          if @FULL_CALLBACK[data.id]
            for key, callback of @FULL_CALLBACK[data.id]
              callback( @_fillUp( App[@className].find(data.id) ) )
              delete @FULL_CALLBACK[data.id][key]
            if _.isEmpty @FULL_CALLBACK[data.id]
              delete @FULL_CALLBACK[data.id]

        error: (xhr, statusText, error) =>
          @FULL_FETCH[id] = false
          App.Log.error('Model', statusText, error, url)
      )
    subscribeId

  ###

  methodWhichIsCalledAtLocalOrServerSiteChange = (changedItems) ->
    console.log("Collection has changed", changedItems, localOrServer)

  # localOrServer can be:
  #  change -> has changed local
  #  destroy -> has been removed local or remote
  #  refresh -> has been changed remote

  params =
    initFetch: true # fetch initial collection

  @subscribeId = App.Model.subscribe(methodWhichIsCalledAtLocalOrServerSiteChange)

  ###

  @subscribe: (callback, param = {}) ->

    # global bind to changes
    if !@SUBSCRIPTION_COLLECTION
      @SUBSCRIPTION_COLLECTION = {}

      # subscribe and render data / fetch new data if triggered
      @bind(
        'refresh'
        (items, options) =>
          if !_.isArray(items)
            items = [items]
          App.Log.debug('Model', "local collection refresh #{@className}", items)
          for key, callback of @SUBSCRIPTION_COLLECTION
            callback(items, 'refresh')
      )
      @bind(
        'change'
        (items, subEvent) =>
          return if subEvent is 'destroy'
          if !_.isArray(items)
            items = [items]
          App.Log.debug('Model', "local collection change #{@className}", items)
          for key, callback of @SUBSCRIPTION_COLLECTION
            callback(items, 'change')
      )
      @bind(
        'destroy'
        (items) =>
          if !_.isArray(items)
            items = [items]
          App.Log.debug('Model', "local collection destroy #{@className}", items)
          for key, callback of @SUBSCRIPTION_COLLECTION
            callback(items, 'destroy')
      )

      # fetch() all on network notify
      events = "#{@className}:create #{@className}:update #{@className}:touch #{@className}:destroy"
      App.Event.bind(
        events
        =>
          return if _.isEmpty(@SUBSCRIPTION_COLLECTION)
          App.Log.debug('Model', "server notify collection change #{@className}")
          callback = =>
            @fetchFull(
              ->
              clear: true
            )
          App.Delay.set(callback, 200, "fullcollection-#{@className}", "model-#{@className}")

        "Collection::Subscribe::#{@className}"
      )

    key = "#{@className}-#{Math.floor(Math.random() * 99999)}"
    @SUBSCRIPTION_COLLECTION[key] = callback

    # fetch init collection
    if param.initFetch is true
      clear = true
      if param.clear is true || param.clear is false
        clear = param.clear
      if !@initFetchActives && @count() is 0
        @initFetchActive = true
        @one('refresh', (collection) =>
          @initFetchActive = false
          callback(collection)
        )
        @fetchFull(
          ->
          clear: clear
        )
      else
        callback(@all())

    # return key
    key

  ###

  methodWhichIsCalledAtLocalOrServerSiteChange = (changedItem, localOrServer) ->
    console.log("Item has changed", changedItem, localOrServer)

  # localOrServer can be:
  #  change -> has changed local
  #  destroy -> has been removed local or remote
  #  refresh -> has been changed remote

  model = App.Model.find(1)
  @subscribeId = model.subscribe(methodWhichIsCalledAtLocalOrServerSiteChange)

  ###

  subscribe: (callback, type) ->

    # remember record id and callback
    App[ @constructor.className ].subscribeItem(@id, callback)

  unsubscribe: (key) ->
    App[ @constructor.className ].unsubscribeItem(@id, key)

  @subscribeItem: (id, callback) ->

    # init bind
    if !@_subscribeItemBindDone
      @_subscribeItemBindDone = true

      # subscribe and render data after local change
      @bind(
        'change'
        (items) =>

          # check if result is array or single item
          if !_.isArray(items)
            items = [items]
          App.Log.debug('Model', "local change #{@className}", items)
          for item in items
            for key, callback of App[@className].SUBSCRIPTION_ITEM[ item.id ]
              callback(App[@className]._fillUp(item), 'change')
      )
      @bind(
        'destroy'
        (items) =>

          # check if result is array or single item
          if !_.isArray(items)
            items = [items]
          App.Log.debug('Model', "local destroy #{@className}", items)
          for item in items
            for key, callback of App[@className].SUBSCRIPTION_ITEM[ item.id ]
              callback(App[@className]._fillUp(item), 'destroy')
      )

      @changeTable = {}
      @bind(
        'refresh'
        (items) =>

          # check if result is array or single item
          if !_.isArray(items)
            items = [items]
          App.Log.debug('Model', "local refresh #{@className}", items)
          for item in items
            for key, callback of App[@className].SUBSCRIPTION_ITEM[ item.id ]

              # only trigger callbacks if object has changed
              if !@changeTable[key] || @changeTable[key] < item.updated_at
                @changeTable[key] = item.updated_at
                callback(App[@className]._fillUp(item), 'refresh')
      )

      # subscribe and render data after server change
      events = "#{@className}:create #{@className}:update #{@className}:touch"
      App.Event.bind(
        events
        (item) =>
          return if !@SUBSCRIPTION_ITEM || !@SUBSCRIPTION_ITEM[ item.id ]
          App.Log.debug('Model', "server change on #{@className}.find(#{item.id}) #{item.updated_at}")
          callback = =>
            genericObject = undefined
            if App[@className].exists(item.id)
              genericObject = App[@className].find(item.id)
            if !genericObject || new Date(item.updated_at) > new Date(genericObject.updated_at)
              App.Log.debug('Model', "request #{@className}.find(#{item.id}) from server")
              @full(item.id, false, true)
          App.Delay.set(callback, 600, "full-#{item.id}", "model-#{@className}")
        "Item::Subscribe::#{@className}"
      )

      events = "#{@className}:destroy"
      App.Event.bind(
        events
        (item) =>
          return if !@SUBSCRIPTION_ITEM || !@SUBSCRIPTION_ITEM[ item.id ]
          return if !App[@className].exists(item.id)
          genericObject = App[@className].find(item.id)
          App.Log.debug('Model', "server delete on #{@className}.find(#{item.id}) #{item.updated_at}")
          callback = ->
            genericObject.trigger('destroy', genericObject)
          App.Delay.set(callback, 500, "delete-#{item.id}", "model-#{@className}")
        "Item::SubscribeDelete::#{@className}"
      )

    # remember item callback
    if !@SUBSCRIPTION_ITEM
      @SUBSCRIPTION_ITEM = {}
    if !@SUBSCRIPTION_ITEM[id]
      @SUBSCRIPTION_ITEM[id] = {}
    key = @className + '-' + Math.floor( Math.random() * 99999 )
    @SUBSCRIPTION_ITEM[id][key] = callback
    key

  @unsubscribeItem: (id, key) ->
    return if !@SUBSCRIPTION_ITEM
    return if !@SUBSCRIPTION_ITEM[id]
    return if !@SUBSCRIPTION_ITEM[id][key]

    delete @SUBSCRIPTION_ITEM[id][key]

  ###

  unsubscribe from model or collection

  App.Model.unsubscribe(@subscribeId)

  ###

  @unsubscribe: (subscribeId) ->
    if @SUBSCRIPTION_ITEM
      for id, keys of @SUBSCRIPTION_ITEM
        if keys[subscribeId]
          delete keys[subscribeId]

    if @SUBSCRIPTION_COLLECTION
      if @SUBSCRIPTION_COLLECTION[subscribeId]
        delete @SUBSCRIPTION_COLLECTION[subscribeId]

  ###

  fetch full collection (with assets)

  App.Model.fetchFull(@callback)

  App.Model.fetchFull(
    @callback
    clear: true
    force: false # only do server call if no record exists
  )


  ###
  @fetchFull: (callback, params = {}) ->
    url = "#{@url}/?full=true"
    App.Log.debug('Model', "fetchFull collection #{@className}", url)

    # request already active, queue callback
    queueManagerName = "#{@className}::fetchFull"
    if params.force is false && App[@className].count() isnt 0
      if callback
        localCallback = =>
          callback(App[@className].all(), 'full')
        App.QueueManager.add(queueManagerName, localCallback)
        App.QueueManager.run(queueManagerName)
      return

    if callback
      localCallback = =>
        callback(App[@className].all())
      App.QueueManager.add(queueManagerName, localCallback)

    return if @fetchFullActive && @fetchFullActive > new Date().getTime() - 500
    @fetchFullActive = new Date().getTime()
    App.Ajax.request(
      type:  'GET'
      url:   url
      processData: true,
      success: (data, status, xhr) =>
        @fetchFullActive = false

        App.Log.debug('Model', "got fetchFull collection #{@className}", data)

        # clear collection
        if params.clear
          App[@className].deleteAll()

        # full / load assets
        if data.assets
          App.Collection.loadAssets(data.assets, targetModel: @className)

          # if no record_ids are found, no initial render is fired
          if data.record_ids && _.isEmpty(data.record_ids)
            App[@className].trigger('refresh', [])

        # find / load object
        else
          App[@className].refresh(data)

        App.QueueManager.run(queueManagerName)

      error: (xhr, statusText, error) =>
        @fetchFullActive = false
        App.Log.error('Model', statusText, error, url)
    )

  ###

  index full collection (with assets)

  App.Model.indexFull(@callback)

  App.Model.indexFull(
    @callback
    page: 1
    per_page: 10
    sort_by: 'name'
    order_by: 'ASC'
  )


  ###
  @indexFull: (callback, params = {}) ->
    url = "#{@url}?full=true"
    for key in ['page', 'per_page', 'sort_by', 'order_by']
      continue if !params[key]
      url += "&#{key}=#{params[key]}"

    App.Log.debug('Model', "indexFull collection #{@className}", url)

    # request already active, queue callback
    queueManagerName = "#{@className}::indexFull"

    if params.refresh is undefined
      params.refresh = true

    App.Ajax.request(
      type:  'GET'
      url:   url
      processData: true,
      success: (data, status, xhr) =>
        App.Log.debug('Model', "got indexFull collection #{@className}", data)

        recordIds = data.record_ids
        if data.record_ids is undefined
          recordIds = data[ @className.toLowerCase() + '_ids' ]

        # full / load assets
        if data.assets
          App.Collection.loadAssets(data.assets, targetModel: @className)

          # if no record_ids are found, no initial render is fired
          if data.record_ids && _.isEmpty(data.record_ids) && params.refresh
            App[@className].trigger('refresh', [])

        # find / load object
        else if params.refresh
          App[@className].refresh(data)

        if callback
          localCallback = =>
            collection = []
            for id in recordIds
              collection.push App[@className].find(id)
            callback(collection, data)
          App.QueueManager.add(queueManagerName, localCallback)

        App.QueueManager.run(queueManagerName)

      error: (xhr, statusText, error) =>
        @indexFullActive = false
        App.Log.error('Model', statusText, error, url)
    )

  ###

  search full collection (with assets)

  App.Model.searchFull(@callback)

  App.Model.searchFull(
    @callback
    query: 'search string'
    page: 1
    per_page: 10
    sort_by: 'name'
    order_by: 'ASC'
  )


  ###
  @searchFull: (callback, params = {}) ->
    if params.full is undefined
      params.full = true

    url = "#{@url}/search"
    App.Log.debug('Model', "searchFull collection #{@className}", url)

    # request already active, queue callback
    queueManagerName = "#{@className}::searchFull"

    if params.refresh is undefined
      params.refresh = true

    App.Ajax.request(
      type:  'POST'
      url:   url
      processData: true,
      data: JSON.stringify(params)
      success: (data, status, xhr) =>
        App.Log.debug('Model', "got searchFull collection #{@className}", data)

        recordIds = data.record_ids

        # full / load assets
        if data.assets
          App.Collection.loadAssets(data.assets, targetModel: @className)

          # if no record_ids are found, no initial render is fired
          if data.record_ids && _.isEmpty(data.record_ids) && params.refresh
            App[@className].trigger('refresh', [])

        # find / load object
        else if params.refresh
          App[@className].refresh(data)

        if callback
          localCallback = =>
            collection = []
            for id in recordIds
              collection.push App[@className].find(id)
            callback(collection, data)
          App.QueueManager.add(queueManagerName, localCallback)

        App.QueueManager.run(queueManagerName)

      error: (xhr, statusText, error) =>
        @searchFullActive = false
        App.Log.error('Model', statusText, error, url)
    )

  @_bindsEmpty: ->
    if @SUBSCRIPTION_ITEM
      for id, keys of @SUBSCRIPTION_ITEM
        return false if !_.isEmpty(keys)

    if @SUBSCRIPTION_COLLECTION && !_.isEmpty(@SUBSCRIPTION_COLLECTION)
      return false

    return true

  @_fillUp: (data, classNames = []) ->

    # fill up via relations
    return data if !data
    return data if !App[@className].configure_attributes
    for attribute in App[@className].configure_attributes

      # lookup relations
      if attribute.relation

        # relations - not calling object itself, to prevent loops
        if !_.contains(classNames, @className)

          # only if relation model exists
          if App[attribute.relation]
            withoutId = attribute.name.substr(0, attribute.name.length - 3)
            if attribute.name.substr(attribute.name.length - 3, attribute.name.length) is '_id'
              if data[attribute.name]

                # only if relation record exists in collection
                if App[ attribute.relation ].exists(data[attribute.name])
                  item = App[attribute.relation].findNative(data[attribute.name])
                  item = App[attribute.relation]._fillUp(item, classNames.concat(@className))
                  data[withoutId] = item
                else
                  if !attribute.do_not_log
                    console.log("ERROR, cant find #{attribute.name} App.#{attribute.relation}.find(#{data[attribute.name]}) for '#{data.constructor.className}' #{data.displayName()}")
    data

  ###

    result = App.Model.search(
      sortBy: 'name'
      order:  'DESC' # default is ASC

      # just show this values in result, all filters need to match to get shown
      filter:

        # check single value
        some_attribute1: 'only_this_value1'

      # just show this values in result, all filters need to match to get shown
      filterExtended:
        [
          some_attribute1: 'regex_to_match1'
          some_attribute2: 'regex_to_match2'
        ]
    )

    returns:

      [ array of objects ]

    examples:

      list = App.Model.search({sortBy:'updated_at', order:'DESC'})

  ###

  @search: (params) ->
    all = @all()
    all_complied = []
    if !params
      for item in all
        item_new = @findNative(item.id)
        all_complied.push @_fillUp(item_new)
      return all_complied
    for item in all
      item_new = @findNative(item.id)
      all_complied.push @_fillUp(item_new)

    # filter search
    if params.filter
      all_complied = @_filter(all_complied, params.filter)

    # use extend filter search
    if params.filterExtended
      all_complied = @_filterExtended(all_complied, params.filterExtended)

    # sort by
    # if translate true then use translated strings to sort list
    if params.sortBy != null
      all_complied = @_sortBy(all_complied, params.sortBy, params.translate)

    # order
    if params.order
      all_complied = @_order(all_complied, params.order)

    all_complied

  @_sortByItem: (item, attribute, translate) ->
    # set displayName as default sort attribute
    if !attribute
      attribute = 'displayName'

    # check if displayName exists
    if attribute is 'displayName'
      if item.displayName
        value = item.displayName()
        valueProcessed = if translate then App.i18n.translateInline(value) else value
        return valueProcessed.toLowerCase()
      else
        attribute = 'name'

    return '' if item[ attribute ] is undefined
    return '' if item[ attribute ] is null

    # return value if string
    if item[ attribute ].toLowerCase
      value = item[ attribute ]
      valueProcessed = if translate then App.i18n.translateInline(value) else value
      return valueProcessed.toLowerCase()

    item[ attribute ]

  @_sortBy: (collection, attribute, translate) ->
    collection.sort (a,b) =>
      aValue = @_sortByItem(a, attribute, translate)
      bValue = @_sortByItem(b, attribute, translate)

      return aValue.localeCompare(bValue) if aValue.localeCompare

      if aValue > bValue || aValue is null
        1
      else if aValue < bValue || bValue is null
        -1
      else
        0

  @_order: (collection, attribute) ->
    if attribute is 'DESC'
      return collection.reverse()
    collection

  @_filter: (collection, filter) ->
    for key, value of filter
      filterValue = value
      if !_.isArray(filterValue)
        filterValue = [filterValue]

      collection = _.filter(collection, (item) ->
        itemValue = item[key]
        if !_.isArray(itemValue)
          itemValue = [itemValue]

        for value in filterValue
          return item if _.contains(itemValue, value)
      )
    collection

  @_filterExtended: (collection, filters) ->
    collection = _.filter(collection, (item) ->

      # check all filters
      for filter in filters

        # all conditions need match
        matchInner = undefined
        for key, value of filter

          if matchInner isnt false
            reg = new RegExp( value, 'i' )
            if item[ key ] isnt undefined && item[ key ] isnt null && item[ key ].match(reg)
              matchInner = true
            else
              matchInner = false

        # if all matched, add item to new collection
        if matchInner is true
          return item

      return
    )
    collection

  activityMessage: (item) ->
    return if !item

    return "Need own activityMessage() in model to generate text (#{@objectDisplayName()}/#{item.type})."

  @lastUpdatedAt: ->
    updated_at
    for item in @all()
      if item.updated_at
        if !updated_at
          updated_at = item.updated_at
        else if item.updated_at > updated_at
          updated_at = item.updated_at
    updated_at

  @updatedAt: (id) ->
    return if !@irecords[id]
    @irecords[id].updated_at

  @findNative: (id) ->
    @irecords[id] or notFound?(id)

  @tagGet: (id, key, callback) ->
    App.Ajax.request(
      id:   key
      type: 'GET'
      url:  "#{@apiPath}/tags"
      data:
        object: @serverClassName || @className
        o_id:   id
      processData: true
      success: (data, status, xhr) ->
        callback(data)
    )

  @tagAdd: (id, item) ->
    App.Ajax.request(
      type: 'POST'
      url:  "#{@apiPath}/tags/add"
      data: JSON.stringify
        object: @serverClassName || @className
        o_id:   id
        item:   item
      processData: true
    )

  @tagRemove: (id, item) ->
    App.Ajax.request(
      type: 'DELETE'
      url:  "#{@apiPath}/tags/remove"
      data: JSON.stringify
        object: @serverClassName || @className
        o_id:   id
        item:   item
      processData: true
    )

  @clearInMemory: ->
    App.Delay.clearLevel("model-#{@className}")

    # reset callbacks to session based functions
    @resetCallbacks()

    # reset attributes to prevent cached forms on relogin
    @resetAttributes()

    # reset cached values of model
    @deleteAll()

  @updateAttributes: (attributes) ->
    if _.isEmpty(@org_configure_attributes)

      # use jquery instead of ._clone() because we need a deep copy of the obj
      @org_configure_attributes = $.extend(true, [], @configure_attributes)
    configure_attributes = $.extend(true, [], @configure_attributes)
    allAttributes = []
    for attribute in attributes
      @attributes.push attribute.name

      found = false
      for attribute_model, index in configure_attributes
        continue if attribute_model.name != attribute.name

        allAttributes.push $.extend(true, attribute_model, attribute)
        configure_attributes.splice(index, 1) # remove found attribute

        found = true
        break

      if !found
        allAttributes.push $.extend(true, {}, attribute)

    @configure_attributes = $.extend(true, [], allAttributes.concat(configure_attributes))

  replaceTagsFunctionCallback: (functionName, parameters) ->
    functionMapping = App[ @constructor.className ].allowedReplaceTagsFunctionMapping[functionName]
    return if !functionMapping

    # First check, if there is a defined allowed function mapping inside the single moodel.
    mappedFunctionName = functionMapping.function_name
    return if !mappedFunctionName

    @[mappedFunctionName](parameters...)

  @resetAttributes: ->
    return if _.isEmpty(@org_configure_attributes)

    # use jquery instead of ._clone() because we need a deep copy of the obj
    @configure_attributes = $.extend(true, [], @org_configure_attributes)

  @resetCallbacks: ->
    if @SUBSCRIPTION_ITEM
      @SUBSCRIPTION_ITEM = {}
    if @SUBSCRIPTION_COLLECTION
      @SUBSCRIPTION_COLLECTION = {}
