// Copyright (C) 2012-2025 Zammad Foundation, https://zammad-foundation.org/

import { mockApolloClient } from '#cy/utils.ts'

import { FormUploadCacheAddDocument } from '#shared/components/Form/fields/FieldFile/graphql/mutations/uploadCache/add.api.ts'

import { mountEditor } from './utils.ts'

const testAction = (
  action: string,
  expected: (text: string) => string,
  submenu?: string,
  role = 'textbox',
  typeText = 'Something',
  hint = ' ',
) => {
  describe(`testing action - ${action}`, { retries: 2 }, () => {
    it(`${action}${hint} - enabled, text after is affected`, () => {
      mountEditor()

      cy.findByRole('textbox').click()

      if (submenu) {
        cy.findByLabelText(submenu).click()
        cy.findByLabelText(action).click()
      } else {
        cy.findByTestId('action-bar').findByLabelText(action).click()
      }

      // It is unsafe to chain further commands that rely on the subject after `.type()`.
      //   https://docs.cypress.io/api/commands/type
      cy.findByRole(role).type(typeText)
      cy.findByRole('textbox').shouldContainNormalizedHtml(typeText)
    })

    it(`${action}${hint} - toggle text`, () => {
      mountEditor()

      cy.findByRole('textbox').type(`${typeText}{selectall}`)

      if (submenu) {
        cy.findByLabelText(submenu).click()
        cy.findByLabelText(action).click()
      } else {
        cy.findByTestId('action-bar').findByLabelText(action).click()
      }

      cy.findByRole('textbox').shouldContainNormalizedHtml(typeText)
    })
  })
}

const testTableAction = (
  actionLabel: string,
  { trCount, tdCount, thCount }: { trCount: number; tdCount: number; thCount?: number },
) => {
  it(`table action - ${actionLabel}`, () => {
    mountEditor()

    cy.findByRole('textbox').click()

    cy.findByTestId('action-bar').findByLabelText('Insert table').click()

    cy.findByRole('table').find('td').first().click()
    cy.findByRole('table').find('td').first().click()

    cy.findByRole('presentation').should('exist')

    if (actionLabel === 'Merge cells') {
      // Not testable since we would have to select two cells, but Cypress does not support it
      // cy.findByRole('table').find('td').eq(2).selectText('left', 2)
    }

    if (actionLabel === 'Split cells') {
      cy.findByRole('table').find('td').selectText('left', 2)
      cy.findByLabelText('Merge cells').click({ force: true }) // can be out of viewport scrollable
      cy.findByRole('table').find('td').first().click()
    }

    cy.findByLabelText(actionLabel).click({ force: true }) // can be out of viewport scrollable

    cy.findByRole('table').find('td').should('have.length', tdCount)
    cy.findByRole('table').find('tr').should('have.length', trCount)

    if (thCount) {
      // cy.findByRole('table').find('th').should('have.length', thCount)
    }
  })
}

// Some test examples in this suite may be flaky due to asynchronous nature of the editor typing mechanism.
//   Configure run mode to retry several times before giving up.
describe('testing actions', { retries: { runMode: 2 } }, () => {
  testAction('Format as underlined', (text) => `<u>${text}</u>`)
  testAction('Format as bold', (text) => `<strong>${text}</strong>`)
  testAction('Format as italic', (text) => `<em>${text}</em>`)
  testAction('Format as strikethrough', (text) => `<s>${text}</s>`)
  testAction('Heading 1', (text) => `<h1>${text}</h1>`, 'Add heading', 'heading')
  testAction('Heading 2', (text) => `<h2>${text}</h2>`, 'Add heading', 'heading')
  testAction('Heading 3', (text) => `<h3>${text}</h3>`, 'Add heading', 'heading')
  testAction('Format as quoted text', (text) => `<blockquote><p>${text}</p></blockquote>`)
  testAction(
    'Add ordered list',
    (text) => `<ol><li><p>${text}</p></li></ol>`,
    undefined,
    'listitem',
  )
  testAction('Add bullet list', (text) => `<ul><li><p>${text}</p></li></ul>`, undefined, 'listitem')

  testAction(
    'Add ordered list',
    () => `<ol><li><p>Something1</p></li></ol>`,
    undefined,
    'listitem',
    'Something1',
    ' (multiline)',
  )
  testAction(
    'Add bullet list',
    () => `<ul><li><p>Something1</p></li></ul>`,
    undefined,
    'listitem',
    'Something1',
    ' (multiline)',
  )

  describe('testing action - remove formatting', () => {
    it('removes formatting', () => {
      mountEditor()

      cy.findByRole('textbox').click()
      cy.findByTestId('action-bar').findByLabelText('Format as bold').click()
      cy.findByRole('paragraph').type('Text')

      cy.findByRole('textbox').shouldContainNormalizedHtml('<p><strong>Text</strong></p>')

      cy.findByRole('textbox').type('{selectall}')
      cy.findByTestId('action-bar').findByLabelText('Remove formatting').click()
      cy.findByRole('textbox').shouldContainNormalizedHtml('<p>Text</p>')
    })
  })

  it('adds a link', () => {
    cy.window().then((win) => {
      cy.stub(win, 'prompt').returns('https://example.com')
      mountEditor()
      cy.findByRole('textbox').click()
      cy.findByTestId('action-bar').findByLabelText('Add link').click()
      cy.findByRole('textbox')
        .find('a')
        .should('have.attr', 'href', 'https://example.com')
        .should('have.text', 'https://example.com')
    })
  })

  it('makes text into a link', () => {
    cy.window().then((win) => {
      cy.stub(win, 'prompt').returns('https://example.com')

      mountEditor()

      cy.findByRole('textbox').click().type('Text{selectAll}')
      cy.findByTestId('action-bar').findByLabelText('Add link').click()
      cy.findByRole('textbox')
        .find('a')
        .should('have.attr', 'href', 'https://example.com')
        .should('have.text', 'Text')
    })
  })

  it('inline image', () => {
    const client = mockApolloClient()
    client.setRequestHandler(FormUploadCacheAddDocument, async () => ({
      data: {
        formUploadCacheAdd: {
          __typename: 'FormUploadCacheAddPayload',
          uploadedFiles: [
            {
              __typename: 'StoredFile',
              id: 'gid://zammad/Store/2062',
              name: 'file.png',
              size: 12393,
              type: 'image/png',
            },
          ],
        },
      },
    }))

    cy.intercept('GET', '/api/v1/attachments/2062', { fixture: 'example.png' })

    mountEditor()

    const imageBuffer = Cypress.Buffer.from('some image')

    cy.findByRole('textbox').click()
    cy.findByTestId('action-bar')
      .findByLabelText('Add image')
      .click() // click inserts input into DOM
      .then(() => {
        cy.findByTestId('editor-image-input').selectFile(
          {
            contents: imageBuffer,
            fileName: 'file.png',
            mimeType: 'image/png',
            lastModified: Date.now(),
          },
          { force: true },
        )
      })

    cy.findByRole('textbox').find('img').should('have.attr', 'src', '/api/v1/attachments/2062')
  })

  describe('table', () => {
    it('inserts a table', () => {
      mountEditor()

      cy.findByRole('textbox').click()
      cy.findByTestId('action-bar').findByLabelText('Insert table').click()

      cy.findByRole('table').should('exist')

      cy.findByRole('table').children().should('have.length', 2)

      cy.findByRole('table').find('td').first().type('Table text')

      cy.findByText('Table text').should('exist')

      cy.findByRole('table').find('tr').should('have.length', 3)
    })

    describe('actions', () => {
      testTableAction('Insert row above', { trCount: 4, tdCount: 9 })
      testTableAction('Insert row below', { trCount: 4, tdCount: 9 })
      testTableAction('Delete row', { trCount: 2, tdCount: 3 })
      testTableAction('Insert column before', { trCount: 3, tdCount: 8 })
      testTableAction('Insert column after', { trCount: 3, tdCount: 8 })
      testTableAction('Delete column', { trCount: 3, tdCount: 4 })
      testTableAction('Merge cells', { trCount: 3, tdCount: 6, thCount: 2 })
      testTableAction('Split cells', { trCount: 3, tdCount: 6, thCount: 3 })

      testTableAction('Toggle header row', { trCount: 3, tdCount: 9 })
      testTableAction('Toggle header column', {
        trCount: 3,
        tdCount: 4,
        thCount: 5,
      })
      testTableAction('Toggle header cell', {
        trCount: 3,
        tdCount: 5,
        thCount: 4,
      })

      it('table action - delete table', () => {
        mountEditor()

        cy.findByRole('textbox').click()
        cy.findByTestId('action-bar').findByLabelText('Insert table').click()

        cy.findByRole('table').should('exist')

        cy.findByRole('table').find('td').first().click()
        cy.findByRole('table').find('td').first().click()

        cy.findByRole('presentation').should('exist')

        cy.findByLabelText('Delete table').click({ force: true }) // can be out of viewport scrollable

        cy.findByRole('table').should('not.exist')
      })
    })
  })

  it('should insert code block', () => {
    mountEditor()

    cy.findByRole('textbox').click()
    cy.findByTestId('action-bar').findByLabelText('Insert code block').click()

    cy.findByRole('code').should('exist')

    cy.findByRole('code').type('const vue = "awesome"')

    cy.findByRole('textbox').shouldContainNormalizedHtml(
      '<pre><code>const <span class="hljs-attr">vue</span> = <span class="hljs-string">"awesome"</span></code></pre>',
    )
  })

  it('indents list item', () => {
    mountEditor()

    cy.findByRole('textbox').click()
    cy.findByTestId('action-bar').findByLabelText('Add bullet list').click()

    cy.findByRole('listitem').type('First{enter}Second{enter}Third')

    cy.findByLabelText('Indent text').click()

    cy.findByRole('textbox').shouldContainNormalizedHtml(
      '<li style="margin-left: 1rem"><p>Third</p></li>',
    )

    cy.findByLabelText('Indent text').click()
    cy.findByRole('textbox').shouldContainNormalizedHtml(
      '<li style="margin-left: 2rem"><p>Third</p></li>',
    )
  })

  it('outdents list item', () => {
    mountEditor()

    cy.findByRole('textbox').click()
    cy.findByTestId('action-bar').findByLabelText('Add bullet list').click()

    cy.findByRole('listitem').type('First{enter}Second{enter}Third')

    cy.findByLabelText('Indent text').click()

    cy.findByRole('textbox').shouldContainNormalizedHtml(
      '<li style="margin-left: 1rem"><p>Third</p></li>',
    )

    cy.findByLabelText('Outdent text').click()
    cy.findByRole('textbox').shouldContainNormalizedHtml('<li><p>Third</p></li>')
  })

  it('changes text color of item', () => {
    mountEditor()

    cy.findByRole('textbox').type('world')
    cy.findByRole('textbox').selectText('left', 5)

    cy.findByTestId('action-bar').findByLabelText('Change text color').click()

    cy.get('[data-test-id="popupWindow"]').within(() =>
      cy.get('[style="background-color: rgb(239, 68, 68);"]').click(),
    )

    cy.findByRole('textbox').shouldContainNormalizedHtml(
      '<span style="color: rgb(239, 68, 68)">world</span>',
    )
  })
})
