# Copyright (C) 2012-2025 Zammad Foundation, https://zammad-foundation.org/

FactoryBot.define do
  factory 'knowledge_base/answer/translation/content', aliases: %i[knowledge_base_answer_translation_content] do
    translation { nil }
    body        { Faker::Lorem.paragraph(sentence_count: 3) }

    before(:create) do |content|
      if content.translation.nil?
        create(:knowledge_base_answer_translation, content: content)
      end
    end

    trait :with_video do
      body { '( widget: video, provider: youtube, id: vTTzwJsHpU8 )' }
    end

    trait :with_image do
      body do
        '<img style="width: 152px; max-width: 100%;" src="data:image/jpeg;base64,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">'
      end
    end
  end
end
